#!/usr/bin/php
<?
require "build-update.config";
set_time_limit(0);

checkAuth($auth);

$approve = false;
$suspend = null;
$forceMd5 = false;
$dryRun = false;
$forceDeploy = false;
$Changes = 0;
$ChangeLog = array();
$Installer = array();

$params = array();
for ($i = 0; $i < $_SERVER['argc']; $i++)
	$params[] = strtolower($_SERVER['argv'][$i]);

parseParams($params);

if (!$package)
{
	echo "Usage: " . $_SERVER["argv"][0] ." <package-name> [parameters]\n\n";
	echo "Package name may appear before, between, or after any parameters.\n\n";
	echo "Type /help for more information.\n\n";
	die;
}

$starttime = time();
RunBuild($package);
$endtime = time();

echo "\n\n============================================\n";

if ($dryRun) {
	showChangeLog();
	echo "No revision was written. This was a dry run.\n";
} else {
	if ($Changes == 0)
	{
		echo "NOTE: Not writing a new revision.\n";
		echo "No changes were made since the last revision!\n";
	} else {
		showChangeLog();
		if (!$forceDeploy) {
			while (true)
			{
				echo "\nDo you want to continue with this revision? (Y/N): ";
				$forceDeploy = readConsole();
		
				if (convertToBoolean($forceDeploy))
					break;
			}
		}
		
		if ($forceDeploy) {
			echo "Writing revisions...\n";
			WriteInstallFiles($package);
		
			SendNewRevision($package);
		
			if ($approve)
				ApproveRevision($package);
		} else {
			echo "Revision cancelled.\n";
		}
	}
}

echo "============================================\n\n";

ShowTimeTaken($endtime - $starttime);

function showTimeTaken($diff)
{
	global $package, $Changes, $dryRun, $forceDeploy;

	$secs = $diff % 60;
	$mins = ($diff - $secs) / 60;
	if ($secs < 10)
		$secs = '0' . $secs;
		
	if (!$dryRun && $forceDeploy)
		echo "\nFinished updating package: $package [$Changes instruction revisions]\n";

	echo "Time taken: $mins:$secs minutes\n";
}

function convertToBoolean(&$string)
{
	$test = strtolower(substr($string, 0, 1));
	
	if (in_array($test, array ('y', 'n')))
	{
		$string = ($test == 'y' ? true : false);
		return true;
	} else {
		return false;
	}
}

function readConsole ($length = 5)
{
   if (!isset ($GLOBALS['StdinPointer']))
     $GLOBALS['StdinPointer'] = fopen ("php://stdin","r");

   $line = fgets ($GLOBALS['StdinPointer'], $length);
   return trim ($line);
}

// parse_str seems contrary in regards to magic quotes
function MMparse_str($s, &$result)
{
	$result = array();
	$parts = explode("&", $s);
	foreach ($parts as $part)
	{
		list ($k,$v) = explode("=", $part, 2);
		$result[urldecode($k)] = urldecode($v);
	}
}

function addChangeLog($message, $type, $OS)
{
	global $ChangeLog;

	if (!isset($ChangeLog[$OS][$type]))
		$ChangeLog[$OS][$type] = array();

	$ChangeLog[$OS][$type][] = $message;

	echo "$message\n";
}

function getOperSystems()
{
	return array('linux', 'win32');
}

function showChangeLog()
{
	global $ChangeLog, $Changes;

	$types = array('add' => '+', 'update' => '*', 'delete' => '-', 'op' => 'o');
	$opersystems = getOperSystems();

	echo "Revision Change Log:\n";
	echo "--------------------\n\n";
	
	if ($Changes == 0) {
		echo "No changes were made in this revision.\n";
		return;
	}
	
	foreach($opersystems As $os) {
		echo "Revision changes for $os:\n";
		
		$log = '';

		foreach($types as $type => $char) {
			if (!isset($ChangeLog[$os][$type]))
				continue;
	
			$log .= " " . ucfirst($type) . " Operations (" . sizeof($ChangeLog[$os][$type]) . "):\n";
			
			foreach($ChangeLog[$os][$type] As $change)
				$log .= "  $char $change\n";
		}

		if ($log == '')
			echo " None\n";
		else
			echo $log;

		echo "\n";
	}
}

function ApproveRevision($package)
{
	global $Revision, $auth, $suspend;

	echo "Approving Revision...\n";
	
	$ignoreWarnings = false;
	$updateParentPackages = true;
	$retry = false;

	foreach ($auth As $domain)
	{
		do {
			$client = getSoapClient($domain);
			if (!$client)
				continue;

			$params = array();
			$params['domainLogin'] = $domain['login'];
			$params['domainPassword'] = $domain['password'];
			$params['packageUniqueName'] = $package;
			$params['suspend'] = $suspend;
			$params['ignoreWarnings'] = $ignoreWarnings;
			$params['updateParentPackages'] = $updateParentPackages;
			
			$response = $client->__soapCall('ApprovePackageRevision', array('parameters' => $params));
			$result = $response->ApprovePackageRevisionResult;
			$retry = false;
	
			echo $domain['login'] . ": [" . $result->PackageApprovalResponseCode . "] " . $result->Description . "\n";
			
			if (!$result->Success) {
				switch($result->PackageApprovalErrorCode) {
					case "None":
						break;
						
					case "UnapprovedParentPackages":
						$updateParentPackages = false;
						while (true) {
							echo "\nShould the parent packages for " . $package . " be approved (and " . ($suspend ? "have servers suspended" : "not suspend servers") . ")? (Y/N): ";
							$updateParentPackages = readConsole();
					
							if (convertToBoolean($updateParentPackages))
								break;
						}
						$ignoreWarnings = true;
						$retry = true;

						break;
					default:
						echo "Unknown response error code: " . $result->PackageApprovalErrorCode . "\n";
						break;
				}
			}
		} while ($retry);

		$ignoreWarnings = false;
	}
}

function SendNewRevision($package)
{
	global $Revision, $auth;

	echo "Notifying GameCreate of new revision...\n";

	foreach ($auth As $domain)
	{
		$client = getSoapClient($domain);
		if (!$client)
			continue;

		$params = array();
		$params['domainLogin'] = $domain['login'];
		$params['domainPassword'] = $domain['password'];
		$params['packageUniqueName'] = $package;
		$params['revision'] = $Revision;

		$response = $client->__soapCall('NotifyPackageRevision', array('parameters' => $params));
		$result = $response->NotifyPackageRevisionResult;

		if (!$result) {
			echo "Warning: An error occurred in notifying GameCreate about the new package revision for domain login " . $domain['login'] . "\nIs your domain login information correct and does the domain package exist?\n";
		}
	}
}

function getSoapClient($domain)
{
	static $client = array();
	$id = $domain['region'];

	if (!isset($client[$id])) {
		$urlPrefix = (substr($id, 0, 7) == 'http://') ? $id : 'http://' . $id . '.gamecreate.com';
		$clientUrl = $urlPrefix . '/admin/Remote.asmx?wsdl';
		$client = null;
		
		try {
			$client[$id] = @new SoapClient($clientUrl);
		} catch (SoapFault $ex) {
			$client[$id] = null;
			echo "Warning: Failed to retrieve a valid SOAP Client for $clientUrl\n";
			echo "Reason: " . $ex->getMessage() . "\n";
		}
	}
	
	return $client[$id];	
}

function RunBuild($package)
{
	global $Revision;

	$Revision = FindNextGlobalRevisionForPackage($package);
	$opersystems = getOperSystems();
	
	foreach($opersystems As $os)
		BuildUpdate($os,$package);
}	

function BuildUpdate($OS,$package)
{
	global $Revision, $Files, $FilesIndex, $FileExists, $IgnoreFiles, $IgnoreDirs, $PreviousOperations, $Operations, $OperationsIndex;

	// Empty out our files information
	$Files = $FilesIndex = $FileExists = $IgnoreFiles = $IgnoreDirs = $PreviousOperations = $Operations = $OperationsIndex = array();

	echo "
============================================
Building new installation script for $OS
============================================
\n";

	$directory = GetDirectoryName($package);
	ParsePackagesList($OS,$package, $directory);
	ParseInstallFile($OS,$package);
	CheckForUpdates(CONTENT_ROOT . "$OS/$directory","", $OS);
	CheckForDeletes($OS);
	CheckForOperations($OS, $package);
	PrepareInstallFile($directory, CONTENT_ROOT . "$OS/$package");
}

function PrepareInstallFile($package, $path)
{
	global $Files, $Revision, $Installer;
	echo "Preparing " . (defined('CONTENT_ROOT') ? substr($path, strlen(CONTENT_ROOT)) : 0) . " revision $Revision\n";
	$data = "INSTALLER $package ".sizeof($Files)."\n";
	$data .= join("", $Files);

	$Installer[$path] = $data;
}


function WriteInstallFiles($package)
{
	global $Files, $Revision, $Installer;
	
	$backup = defined('BACKUP_REVISION') && BACKUP_REVISION;

	foreach (array_keys($Installer) As $path)
	{	
		echo "Writing " . (defined('CONTENT_ROOT') ? substr($path, strlen(CONTENT_ROOT)) : 0) . " revision $Revision\n";
		
		if ($backup)
			$file = $path.".".$Revision.".txt";
		else
			$file = $path.".txt";

		$fp = @fopen($file, "w");
		
		if ($fp === false)
			die("An error occurred attempting to open $file for write");
		
		fputs($fp, $Installer[$path]);
		fclose($fp);
		
		if ($backup)
			copy ($path.".".$Revision.".txt",$path.".txt");
	}
}

function CheckForDeletes($OS)
{
	global $Files, $FilesIndex, $FileExists, $Revision, $Changes;
	foreach ($FilesIndex as $name => $index)
	{
		if ($OS == "win32")
			$nameCompare = strtolower($name);
		else
			$nameCompare = $name;

		if (isset($FileExists[$nameCompare]))
			continue;

		$parts = explode(" ", $Files[$FilesIndex[$nameCompare]]);

		if ($parts[0] == "DELETE" || $parts[0] == "RMDIR")
			continue;

		$op = "DELETE";
		if ($parts[0] == "MKDIR")
			$op = "RMDIR";

		addChangeLog("Deleting $name", 'delete', $OS);
		$Changes++;

		$Files[$index]="";
		$newIndex = sizeof($Files);
		$Files[$newIndex] = "$op name=".urlencode($name)." $Revision\n";
		$FilesIndex[$name] = $newIndex;
	}
}

function IgnoreFile($file, $isDirectory)
{
	global $IgnoreFiles, $IgnoreDirs, $OwnedDirs;
	
	// If this directory is along the path of one of our package's explicitly owned
	// directories, then do not ignore this directory as there is content inside it we need to check
	if ($isDirectory) {
		foreach($OwnedDirs As $dir) {
			$dir = substr($dir, 1, strlen($file));
			if ($dir == $file)
				return false;
		}
	}
	
	if (isset($IgnoreFiles[$file]))
		return true;

	foreach ($IgnoreDirs as $dir)
	{
		$dir = substr($dir, 1);
		if ($file == $dir)
			return true;
	}

	return false;
}

function CheckForOperations($OS, $package)
{
	global $PreviousOperations, $Operations, $OperationsIndex, $Files;
	global $Revision;
	global $Changes;

	//Get current file operations
	$contents = @file(CONTENT_ROOT . $OS."/".$package.".op.txt");
	if (is_array($contents))
	{
		//Operations are case insensitive in windows
		foreach ($contents As $line)
		{
			$line = trim($line);

			if ($line == "")
				continue;

			if ($OS == "win32")
				$line = strtolower($line);

			$Operations[] = $line;
	
			if (!isset($PreviousOperations[$line]))
			{
	
				//New operation
				addChangeLog("Adding new EXECUTABLE operation for $line", 'op', $OS);
				$Files[] = "EXECUTABLE name=".urlencode($line)." $Revision\n";
				$Changes++;
			}
		}
	}

	foreach ($PreviousOperations As $key => $dummy)
	{
		if (array_search($key, $Operations) !== false)
			continue;

		$index = -1;
		foreach ($Files as $fileValue)
		{
			$index++;
			$parts = explode(" ", $fileValue);

			if ($parts[0] != "EXECUTABLE")
				continue;

			MMparse_str($parts[1], $values);
			
			if ($OS == "win32")
				$values["name"] = strtolower($values["name"]);

			if ($key == $values["name"])
			{
				addChangeLog("Removing operation on file: " . $values["name"], 'op', $OS);
				$Changes++;
				$Files[$index] = '';
			}
		}

		
	}
}

function CheckForUpdates($realDir, $logicalDir, $OS)
{
	global $Files, $FilesIndex, $FileExists, $Changes, $forceMd5;
	global $Revision;

	$handle = @opendir($realDir);
	if (!$handle)
		return;

	while (($file = readdir($handle)) !== FALSE)
	{
		if ($file == "." || $file == "..") continue;
		
		$logicalFile = $logicalDir.$file;

		if ($OS == "linux")
		{
			//Case sensitivity matters in linux files, FILE is different from file.
			$compareFile = $logicalFile;
		} else {
			//Ignore case changes when building updates for other (win32) OSs - FILE is the same as file
			$compareFile = strtolower($logicalFile);
		}

		$realFile = $realDir."/".$file;
		$isDirectory = is_dir($realFile);
		
		// Ignore file owned by other packages
		if (IgnoreFile($compareFile, $isDirectory))
		{
			echo "Ignoring " . ($isDirectory ? "directory" : "file") . " $logicalFile\n";
			continue;
		}

		// Handle directory
		if (is_dir($realFile))
		{
			$FileExists[$compareFile] = true;
			if (!isset($FilesIndex[$compareFile]))
			{
				$index = sizeof($Files);
				$Files[$index] = "MKDIR name=".urlencode($logicalFile)." $Revision\n";
				$FilesIndex[$compareFile] = $index;
				$Changes++;
				addChangeLog("Adding new directory $logicalFile", 'add', $OS);
			}

			echo "Checking $logicalDir$file/ for updates\n";
			CheckForUpdates($realDir."/".$file, "$logicalDir$file/", $OS);
			continue;
		}


		// Mark file as still existing
		$FileExists[$compareFile] = true;

		$time = filemtime($realFile);
		$size = filesize($realFile);
			
		// New file?
		if (!isset($FilesIndex[$compareFile]))
		{
			$md5 = md5_file($realFile);
			$index = sizeof($Files);
			$Files[$index] = "FILE name=".urlencode($logicalFile)."&md5sum=$md5&time=$time&size=$size $Revision\n";
			$FilesIndex[$compareFile] = $index;
			addChangeLog("Adding new file $logicalFile", 'add', $OS);
			$Changes++;
		}
		else
		{

			$index = $FilesIndex[$compareFile];

			// Check if this file's size or time differs. If so, do the more intensive md5sum check
			if (!$forceMd5 && isset($Files[$index]) && GetTimeFromLine($Files[$index]) == $time && GetSizeFromLine($Files[$index]) == $size)
			{
				//echo "Ignoring unmodified file $logicalFile (No time or size change)\n";
				continue;
			}

			$md5 = md5_file($realFile);
			$oldmd5 = GetMd5FromLine($Files[$index]);

			if ($oldmd5 == $md5)
			{
				// md5 does not differ, however the file time/size was different.
				// Update our FILE line with these new values so we dont check the md5 again next run

				$Files[$index] = "FILE name=".urlencode($logicalFile)."&md5sum=$md5&time=$time&size=$size " . GetRevisionFromLine($Files[$index]) . "\n";
				echo "Ignoring unmodified file $logicalFile (No md5sum change)\n";

			}
			else
			{
				$Files[$index] = "";
				$index = sizeof($Files);
				$Files[$index] = "FILE name=".urlencode($logicalFile)."&md5sum=$md5&time=$time&size=$size $Revision\n";
				$FilesIndex[$compareFile] = $index;
				$Changes++;
				addChangeLog("Updating file $logicalFile", 'update', $OS);
			}
		}
	}
}

function GetMd5FromLine($line)
{
	$parts = explode(" ", trim($line));
	MMparse_str($parts[1], $values);
	return @$values["md5sum"];
}

function GetRevisionFromLine($line)
{
	$parts = explode(" ", trim($line));
	return @$parts[2];
}

function GetTimeFromLine($line)
{
	$parts = explode(" ", trim($line));
	MMparse_str($parts[1], $values);
	return @$values["time"];
}

function GetSizeFromLine($line)
{
	$parts = explode(" ", trim($line));
	MMparse_str($parts[1], $values);
	return @$values["size"];
}

function FindNextGlobalRevisionForPackage($package)
{
	$opersystems = getOperSystems();
	
	$NewRevision = 1;
	
	foreach($opersystems As $os) {
		$r = FindNextRevisionForOSAndPackage($os, $package);
		
		if ($r > $NewRevision)
			$NewRevision = $r;
	}
	
	return $NewRevision;
}

function FindNextRevisionForOSAndPackage($OS, $package)
{
	$lines = @file(CONTENT_ROOT . $OS."/".$package.".txt");

	// No revision information - start at Revision 1
	if(sizeof($lines) <= 1)
		return 1;

	// Ignore first line (INSTALLER packagename instructioncount)
	array_shift($lines);

	$revision = 1;

	foreach ($lines As $line) {
		$line = trim($line);
		$bits = explode(" ", $line);
		$rev = $bits[sizeof($bits)-1];

		if ($rev > $revision)
			$revision = $rev;
	}

	return $revision + 1;
}

function ParseInstallFile($OS, $package)
{
	global $Files;
	global $FilesIndex;
	global $Installer;
	global $PreviousOperations, $Operations;

	$Files = $FilesIndex = $PreviousOperations = $Operations = array();

	$contents = @file(CONTENT_ROOT . $OS."/".$package.".txt");
	if (!is_array($contents))
		return;

	array_splice($contents,0,1);

	$Files = $contents;
	for ($i=0,$j=sizeof($contents);$i<$j;$i++)
	{
		$line = $Files[$i];
		$parts = explode(" ", trim($line));

		if (!isset($parts[1]))
			continue;

		MMparse_str($parts[1], $values);

		if ($OS == "win32")
			$values["name"] = strtolower($values["name"]);

		if ($parts[0] == "EXECUTABLE")
		{
			$PreviousOperations[trim($values["name"])] = true;
		} else {
			if ($values["name"])
				$FilesIndex[$values["name"]] = $i;
		}
	}
}


function GetDirectoryName($package)
{
	$contents = @file("packages.txt");
	if ($contents == '')
	{
		echo "Could not find package '$package' - did you add it to packages.txt?\n";
		die;
	}
	
	foreach ($contents as $line)
	{
		$parts = explode(" ", trim($line), 2);
		if ($parts[0] == $package)
			return $parts[1];
	}
	
	echo "Could not find package '$package' - did you add it to packages.txt?\n";
	die;
}

function ParsePackagesList($OS, $package, $directory)
{
	global $IgnoreDirs, $IgnoreFiles, $OwnedDirs;
	$IgnoreDirs = array();
	$IgnoreFiles= array();
	$OwnedDirs = array();

	$contents = file("packages.txt");
	for ($i=0,$j=sizeof($contents);$i<$j;$i++)
	{
		$line = $contents[$i];

		@list ($name,$dir) = explode(" ", trim($line), 2);
		if ($dir != $directory)
			continue;
			
		$isIgnoring = $name != $package;

		if ($isIgnoring)
			AddToIgnoreFiles($OS,$name);

		// We've hit another package sharing our directory
		while ($i+1 < $j && $contents[$i+1][0] == '-')
		{
			$i++;
			$line = $contents[$i];
			list ($dash,$path) = explode(" ", trim($line), 2);

			if ($isIgnoring) {
				$IgnoreDirs[] = ($OS == "win32" ? strtolower($path) : $path);
				echo "Ignoring $name Directory: $path\n";
			} else {
				$OwnedDirs[] = ($OS == "win32" ? strtolower($path) : $path);
			}
		}
	}
}

function AddToIgnoreFiles($OS,$package)
{
	global $IgnoreFiles;

	$contents = @file(CONTENT_ROOT . $OS."/".$package.".txt");
	if (!is_array($contents))
		return;

	foreach ($contents as $line)
	{
		$parts = explode(" ", $line);
		if ($parts[0] != "FILE" && $parts[0] != "MKDIR")
			continue;

		MMparse_str($parts[1], $values);

		if ($OS == "win32")
			$IgnoreFiles[strtolower($values["name"])] = 1;
		else
			$IgnoreFiles[$values["name"]] = 1;
		
	}
}

function showParamHelp()
{
echo "GameCreate Build-Update Parameter Help
======================================
    /approve      - Approve the package revision on a successful version.
    /a

    /restart      - Restart affected servers for this update.
    /r

    /norestart    - Do not restart affected servers for this update.
    /n

    /md5          - Force an md5sum comparison of each file.
                    By default, md5sum comparisons are only performed when a file on disk
                    has a different modified time or file size than what is specified
                    in the previous revision.
                    This option ignores these checks, and will force an md5sum
                    comparison regardless of the file modified time or its size.

    /dry          - Perform a dry run. Changes are displayed but not committed.
    /d

    /confirm      - Automatically confirm the revision changes on success and notify GameCreate.
    /c              No confirmation of the build change log is needed at the end of the build.

    /version      - Show the GameCreate Package Builder version information.
    /v
    
    /help         - This help screen.
";
}

function parseParams(&$params)
{
	global $approve, $suspend, $forceMd5, $dryRun, $forceDeploy, $package;
	
	if (sizeof($params) < 2)
		return;

	for ($i = 1; $i < sizeof($params); $i++)
	{
		switch ($params[$i])
		{
			case "/help": case "/?":
				showParamHelp();
				die;
			case "/approve": case "/a":
				$approve = true;
				break;
			case "/norestart": case "/n":
				$suspend = false;
				break;
			case "/restart": case "/r":
				$suspend = true;
				break;
			case "/dry": case "/d":
				$dryRun = true;
				break;
			case "/confirm": case "/c":
				$forceDeploy = true;
				break;
			case "/md5":
				echo "Ignoring file time and size values; forcing md5sum comparison\n";
				$forceMd5 = true;
				break;
			case "/version": case "/v":
				showVersion();
				die;
			default:
				$unknownParam = false;
				if ($params[$i][0] == '/') {
					$unknownParam = true;
				} else {
					if ($package == '') {
						// Assign package name
						$package = $params[$i];
					} else {
						echo "Package name already specified as '$package'.\n";
						$unknownParam = true;
					}
				}
				
				if ($unknownParam) {
					echo "Unknown parameter: " . $params[$i] . "\n";
					echo "Type /help for a list of supported command arguments.\n";
					die;
				}
				break;
		}
	}
	
	if (!$approve || $suspend !== null)
		return;

	while (true)
	{
		echo "\nShould servers be suspended during this update? (Y/N): ";
		$suspend = readConsole();

		if (convertToBoolean($suspend))
			break;
	}
}

function checkAuth($auth)
{
	if (sizeof($auth) == 0)
		die("You need to define at least one set of domain login information to be used in the build-update.config file.");
	
	$args = array('login', 'password', 'region');
	foreach ($auth As $domain)
	{
		foreach ($args As $arg)
			if (!isset($domain[$arg]) || $domain[$arg] == '')
				die("One or more arguments are undefined for a domain in build-update.config");
	}
}

function showVersion()
{
	echo "GameCreate Package Builder v1.0.2\n";
}

?>
